﻿<#
    #############################################################################################
    .SYNOPSIS
    Copyright(C)2019 FileHold Systems Inc.
    #############################################################################################
#>
$BuildNumber = '20190614.1'

# Load FH API functions. Assume ps1 is in same location as this script.
$ScriptDir = Split-Path -Path $script:MyInvocation.MyCommand.Path
. ( $ScriptDir + '\FileHoldApi.ps1' )

function Add-DropdownValues
{
<#
    #############################################################################################
    .SYNOPSIS
    Add dropdown values to existing metadata field

    .DESCRIPTION
    For a FileHold managed dropdown metadata field, add new values 
    from a Powershell pipeline.

    .PARAMETER Server
    Address of the FileHold server, like http://myserver/fh/filehold
    .PARAMETER UserId
    FileHold user id with a role suitable for this script. If this 
    value and the password are not provided the user will be 
    logged in using integrated authentication.
    .PARAMETER Password
    Password to use with the UserId
    .PARAMETER Domain
    Active directory domain for the user. If this value is not 
    provided the user will be logged in as a local user.

    .NOTES
    Use -WhatIf to perform all functions except updating the field.

    DISCLAIMER. 
    
    FileHold makes no claims to the correctness, fitness for purpose, or
    anything else related to this script. It is provided as an example only.
    It is intended to be used or modified by a person skilled with Windows,
    PowerShell, .NET programming, and the FileHold API. Never use it on a
    production system without thouroughly testing it first and never use it
    in production if you do not fit the skilled person description above.
    #############################################################################################
#>
  [CmdletBinding(SupportsShouldProcess = $true)]
  Param (
    [Parameter(
        ValueFromPipeline = $true,
        HelpMessage='A list of values to add to the dropdown list.',
        Mandatory = $true
    )]
    [Object]$ValueList,
    
    [Parameter(
        Mandatory = $true, 
        HelpMessage='Name or URI for FileHold server. HTTP assumed if protocol not specified.'
    )]
    [uri]$Server,
    
    [Parameter(
        Mandatory = $true,
        HelpMessage = 'Name of FileHold managed dropdown menu to add new values to.'
    )]
    [string]$FieldName,
    
    [string]$UserId = $null,
    
    [string]$Password = $null,
    
    [string]$Domain = $null           
  )
  
  Begin
  {
    Write-Information -MessageData "Import dropdown values $(Get-Date) ($($MyInvocation.MyCommand.Name) $BuildNumber)"

    if ( $WhatIfPreference ) 
    {
      Write-Verbose -Message 'Simulate mode is enabled. The system will not be updated. Invoke without ""-WhatIf"" to make changes.'
    }
    
    if ( $Domain )
    {
      Start-FileHoldSession -HostAddress $Server -UserId $UserId -Password $Password -Domain $Domain -BuildNumber $BuildNumber
    }
    if ( $UserId -and $Password )
    {
      Start-FileHoldSession -HostAddress $Server -UserId $UserId -Password $Password -BuildNumber $BuildNumber
    }
    else
    {
      Start-FileHoldSession -HostAddress $Server -UseIntegratedAuthentication -BuildNumber $BuildNumber 
    }
      
    $metadataFieldsList = (DocumentSchemaManager).GetFieldsOverviewList()
      
    $metadataFieldInfo = $metadataFieldsList.Where( { $_.Name -eq $FieldName } )
    if ( $metadataFieldInfo.Id -eq 0 )                                                     
    {
      Throw "Metadata field ""$FieldName"" does not exist."
    }
    else
    {
      if ( $metadataFieldInfo.FieldType -ne [DocumentSchemaManager.FieldTypes]::DropdownMenu.value__ )
      {
        Throw "Metadata field ""$FieldName"" is not a dropdown menu."
      }
      elseif ( $metadataFieldInfo.IsDBLookup )
      {
        Throw "Dropdown menu metadata field ""$FieldName"" is not FileHold managed."
      }
      else
      {
        $choices = New-Object -TypeName System.Collections.ArrayList -ArgumentList ( $null )
      }
    }      
      
    Write-Information -MessageData "Values will be added to ""$FieldName"" field."
      
    $TotalItems = 0
  }

  Process
  {
    $TotalItems += 1
    foreach ( $Value in $ValueList )
    {
      $field = New-Object -TypeName (LibraryManager.DropdownFieldChoice)
      $field.Value = $Value
      $null = $choices.Add( $field )      
      Write-Information -MessageData "Adding value ""$($Value)"" from record $TotalItems."
    }
  }

  End
  {
    $metadataField = (DocumentSchemaManager).GetGlobalField( $metadataFieldInfo.Id )
    $choices.InsertRange( 0, $metadataField.Choices )
    $metadataField.Choices = $choices
    # The following assignment is required to tell the serializer to not send this parameter to the server.
    # This is required even though the value is $null when the object was created as the WebServiceProxy
    # serializer treats $null as though it is an empty string and SchemaUsedIn is not a writeable field.
    $metadataField.SchemasUsedIn = [NullString]::Value
    if ( $pscmdlet.ShouldProcess( $FieldName, 'Change dropdown choices') )
    {
      (DocumentSchemaManager).ChangeGlobalField( $metadataField, $false )
    }
    Stop-FileHoldSession
  }
}

#Get-Content -Path .\ValueList.txt | Add-DropdownValues -Server RTEST11 -UserId sysadm -Password 12345 -FieldName 'test field' -InformationAction Continue -Verbose
Get-Content -Path .\ValueList.txt | Add-DropdownValues -Server RTEST11 -UserId sysadm -Password 12345 -FieldName 'test field' -WhatIf