﻿<#
.SYNOPSIS
    Create an XML file from an image containing a QR Code barcode with comma delimited fields.

.DESCRIPTION
    The XML format is compatible with the FileHold Desktop Application managed imports feature. 
    
    By default, a log file called 'ConvertQR2Xml<YYYYMMDD>.log' is written to the current 
    folder. The log file for any given day will have new information appended to it each
    time the CmdLet is run.

.NOTES
    Copyright(C)2018 FileHold Systems Inc.

.LINK
    https://www.filehold.com

.PARAMETER Path
    The path to the tree containing the images.

.PARAMETER LogFileBase
    The path and base name used for the log file.

.PARAMETER ImageExtensions
    A filename pattern used to match image files.

.PARAMETER CsvHeader
    An array of field names.

.PARAMETER ZBarDecode
    Path to the Zbar executable for scanning the barcodes. http://zbar.sourceforge.net/

.EXAMPLE
    ConvertQR2Xml -CsvHeader @('Invoice','Vendor','Total')
    Assumes the barcodes contain three fields invoice, vendor, and total.
 #>

Param
( 
    [Parameter( Mandatory=$false)]
    [string]$Path=".\",
    [Parameter( Mandatory=$false)]
    [string]$LogFileBase=".\ConvertQR2Xml",
    [Parameter( Mandatory=$false)]
    [string]$ImageExtensions='*.jpg',
    [Parameter( Mandatory=$false)]
    [string[]]$CsvHeader=@('Fld1', 'Fld2', 'Fld3'),
    [Parameter( Mandatory=$false)]
    [string]$ZBarDecoder='c:\projects\barcode\zbarimg.exe'
)

$global:LogFile = $LogFileBase
$global:logPrepared = $false
$global:errorCount = 0
$global:imageCount = 0
$root = $Path
$global:LogFile += ( Get-Date -Format "yyyyMMdd" ) + ".log"

function StartLogging
{
    try
    {
        "----------------------------------------------------------------------" | Out-File -FilePath $global:LogFile -Append
        "Start processing: " + ( Get-Date -Format "yyyy-MM-dd HH:mm:ss" ) | Out-File -FilePath $global:LogFile -Append
        $global:logPrepared = $true
    }
    catch 
    {
        Write-Host ( "Error during processing. The log file " + $global:logFile + " could not be created or opened." ) -ForegroundColor Red
        Throw 
    }
}

function EndLogging
{
    "End processing: " + ( Get-Date -Format "yyyy-MM-dd HH:mm:ss" ) | Out-File -FilePath $global:LogFile -Append
    "----------------------------------------------------------------------" | Out-File -FilePath $global:LogFile -Append
    $global:logPrepared = $false
}

function Log
{
    Param
    ( 
        [Parameter( Mandatory=$false)]
        [ValidateSet( "Error", "Warning", "Info", "Summary" )]
        [string]$Level="Info", 
        
        [Parameter( Mandatory=$true )]
        [string]$Message
    )

    if ( $global:logPrepared )
    {
        $Level + ":`t" + $Message | Out-File -FilePath $global:LogFile -Append
    }
    if ( $Level -eq "Error" )
    {
        $global:errorCount += 1
    }
}

<#
 # Emit a single metadata field into an existing XML stream.
 #>
function AddField
{
    param( [System.Xml.XmlTextWriter]$XmlWriter, [string]$Name, [string]$Value='' )

    try
    {
        $XmlWriter.WriteStartElement( "Field" )
        $XmlWriter.WriteElementString( "Name", $Name )
        $XmlWriter.WriteElementString( "Value", $Value )
        $XmlWriter.WriteEndElement()
    }
    catch
    {
        Log -Level Error -Message ( "Unable to write the XML file " + $XmlWriter. $XmlName + ". " + $error[0].Exception.Message )
        throw ( "Unable to write the XML file. " + $error[0].Exception.Message )
    }
}


function StartXml( [string]$xmlName )
{
    Log -Message ( "Creating " + $Xmlname )

    try 
    {
        $global:xmlWriter = New-Object System.Xml.XmlTextWriter( $XmlName, $null )
        $global:xmlWriter.Formatting = "Indented"
        $global:xmlWriter.Indentation = 1
        $global:xmlWriter.IndentChar = "`t"
        $global:xmlWriter.WriteStartDocument()
        $global:xmlWriter.WriteStartElement( "Batch" )
    }
    catch
    {       
        Log -Level Error -Message ( "Unable to write the XML file " + $XmlName + ". (StartXml) " + $error[0].Exception.Message )
        throw ( "Unable to write the XML file " + $XmlName + ". (StartXml) " + $error[0].Exception.Message )
    }
}

function EndXml
{
    try
    {
        $global:xmlWriter.WriteEndDocument()
        $global:xmlWriter.Flush()
    }
    catch
    {
        Log -Level Error -Message ( "Unable to finish writing the XML file " + $XmlName + ". (EndXml) " + $error[0].Exception.Message )
        throw ( "Unable to finish writing the XML file " + $XmlName + ". (EndXml) " + $error[0].Exception.Message )
    }
    finally
    {
        if ( $global:xmlWriter ) { $global:xmlWriter.Close() }
    }
}

function CreateXmlPage
{
    param( [System.IO.FileInfo]$image )

    $command = $ZBarDecoder + ' --raw -q ' + $image
    $csvRecord = Invoke-Expression $command | ConvertFrom-Csv -Header $CsvHeader

    if ( $csvRecord.Count -eq 0 )
    {
        Log -Level Warning -Message ( 'No data record for ' + $image + '. No XML file will be created.' )
        $global:noDataCount += 1
    }
    else
    {
        try
        {        
            StartXml (( Join-Path $imageFile.DirectoryName $imageFile.BaseName ) + ".xml" ) -ErrorAction Stop
            $xmlWriter.WriteStartElement( 'Page' )
            AddField -XmlWriter $global:xmlWriter -Name 'File Name' -Value $image -ErrorAction Stop
            foreach ( $fld in ( $csvRecord | Get-Member -MemberType NoteProperty | select -Property Name ))
            {
                AddField -XmlWriter $global:xmlWriter -Name $fld.name -Value $csvRecord.($fld.name) -ErrorAction Stop
            }
            $xmlWriter.WriteEndElement()
        }
        catch
        {
            Log -Level Error -Message ( "Unable to write to the XML file " + $XmlName + ". (CreateXmlPage) " + $error[0].Exception.Message )
            return
        }
        finally
        {
            EndXml -ErrorAction Stop
        }
    }
}

####################################################################################
# 
# Main loop
#
####################################################################################
StartLogging

Log -Message ( "Path to start search: " + $root )
Log -Message ( 'CSV Header: ' + [string]$CsvHeader )
Log -Message ( 'ZBAR: ' + $ZBarDecoder )
#Log -Message ( 'Match field ' + $( if ( $MatchWithExtension ) { 'with ' } else { 'without ' } ) + 'extension' )
Log -Message ( 'Image extensions: ' + $ImageExtensions )

$global:noDataCount = 0
$imageCount = 0
$imageFolderCount = 0
$foldername = ""

$imageFiles = dir -Path $root -Recurse -File $ImageExtensions

$lastFolderName = Split-Path -Path $imageFiles[0].FullName
foreach ( $imageFile in $imageFiles )
{
    $folderName = Split-Path -Path $imageFile.FullName
    if ( $lastFolderName -ne $folderName )
    { 
        Log -Level Summary -Message ( "Processed " + $imageFolderCount + " images in folder " + $lastFolderName + "." )
        $lastFolderName = $folderName
        $imageFoldercount = 0
    }

    $imageCount += 1
    $imageFolderCount += 1
    CreateXmlPage $imageFile.FullName
}

if ( $folderName -ne '' ) { Log -Level Summary -Message ( "Processed " + $imageFolderCount + " images in folder " + $folderName + "." ) }
Log -Level Summary -Message ( "Processed " + $global:imageCount + " images." )
if ( $global:imageCount -gt 0 )
{
    Log -Level Summary -Message ( [string]$global:noDataCount + ' images had no cooresponding data record.' )
}

if ( $errorCount -gt 0 )
{
    Log -Level Summary -Message ( "Errors " + $global:errorCount + "." )
    Write-Host ( "Error during processing. See " + $logFile + " for details." ) -ForegroundColor Red
}

EndLogging
