﻿ <#
.SYNOPSIS

        Change the auto-delegate setting for an arbitrary user

.DESCRIPTION

        The FileHoldApi.psm1 helper module should be in the same folder as this script.
        If you leave out username and password, Windows Integrated Authentication will
        use the credentials of the current user.

.NOTES

        Copyright(C)2022 FileHold Systems Inc.

        *** DISCLAIMER ***

        FileHold makes no claims to the correctness, fitness for purpose, or
        anything else related to this script. It is provided as an example only.
        It is intended to be used or modified by a person skilled with Windows,
        PowerShell, .NET programming, and the FileHold API. Never use it on a
        production system without thouroughly testing it first and never use it
        in production if you do not fit the skilled person description above.

        *** DISCLAIMER ***

.EXAMPLE set new delegate

    Set-FileHoldAutoDelegateUser -Server 'http://myserver' -UserId 'sysadm' -Password '12345' -SearchType ExactName -FromUser 'user one' -ToUser sysadm
     
.EXAMPLE remove delegate

    Set-FileHoldAutoDelegateUser -Server 'http://myserver' -UserId 'sysadm' -Password '12345' -SearchType ExactName -FromUser 'user one'

#>

function Set-FileHoldAutoDelegateUser
{
    [CmdletBinding(SupportsShouldProcess)] # SupportsShouldProcess needed for -WhatIf and -Confirm
    Param (
        [Parameter(HelpMessage='Name or URI for FileHold server. HTTP assumed if protocol not specified.'
        )][uri]$Server = 'localhost',
    
        [Parameter(HelpMessage='FileHold user id with sufficient roll or null for IWA.'
        )][string]$UserId = $null,
    
        [Parameter(HelpMessage='Password for given user or null for IWA.'
        )][string]$Password = $null,
    
        [Parameter(HelpMessage='Active directory domain name for given user.'
        )][string]$Domain = $null,

        [Parameter(HelpMessage='Set to not log off the at the end of the script.'
        )][switch]$PreserveSession = $false,

        [Parameter(HelpMessage='Set to use the current session cache instead of a new log in.'
        )][switch]$ReuseSession= $false,

        [Parameter(HelpMessage='Set this switch to prompt for credentials.'
        )][switch]$PromptForAuthentication = $false,

        [Parameter(HelpMessage='Set this to the method to search the user list.'
        )][ValidateSet( 'ExactName', 'ExactEmail', 'StartsWithName' )]
        [string]$SearchType = 'ExactName',

        [Parameter(HelpMessage='The user whose tasks will be delegated.',Mandatory
        )][string]$FromUser = $null,

        [Parameter(HelpMessage='The user who will be delegated the tasks or null.'
        )][string]$ToUser = $null

    )
  
    Write-Information -MessageData "Set auto-delegation for user $(Get-Date) ($($MyInvocation.MyCommand.Name) $SetFileHoldAutoDelegateUserBuildNumber)"

    if ( $WhatIfPreference ) 
    {
        Write-Verbose -Message 'Simulate mode is enabled. The system will not be updated. Invoke without ""-WhatIf"" to make changes.'
    }
    
    if ( !$ReuseSession )    
    {
        try {
            Start-FileHoldSession    -HostAddress $Server `
                                        -UserId $UserId `
                                        -Password $Password `
                                        -Domain $Domain `
                                        -ClientName "($($MyInvocation.MyCommand.Name) $BuildNumber)" `
                                        -ClientVersion '16.2' `
                                        -InformationAction $InformationPreference `
                                        -PromptForAuthentication:$PromptForAuthentication
        } catch {
            Write-Error $_
            exit
        }
    }      

    try {
        if ( $pscmdlet.ShouldProcess( $FromUser, "Get 'from' user guid" ))
        {
            $fromUserObjects = Get-FileHoldUserObjects -ReuseSession -PreserveSession -ValueList $FromUser
            if ( $fromUserObjects.Count -gt 1 ) { throw "From user '$FromUser' matched more than one user in FileHold." }
            if ( $fromUserObjects.Count -eq 0 ) { throw "From user '$FromUser' not found in FileHold." }
            if ( $fromUserObjects[0].Type -ne 0 -and $fromUserObjects[0].Type -ne 1 ) { throw "From user '$FromUser' is a group." }
            if ( !$fromUserObjects[0].HasFHAccountEnabled ) { throw "From user '$FromUser' is disabled." }
        }
        if ( $ToUser -ne '' )
        {
            if ( $pscmdlet.ShouldProcess( $ToUser, "Get 'to' user guid" ))
            {
                $toUserObjects = Get-FileHoldUserObjects -ReuseSession -PreserveSession -ValueList $ToUser
                if ( $toUserObjects.Count -gt 1 ) { throw "To user '$ToUser' matched more than one user in FileHold." }
                if ( $toUserObjects.Count -eq 0 ) { throw "To user '$ToUser' not found in FileHold." }
                if ( $toUserObjects[0].Type -ne 0 -and $toUserObjects[0].Type -ne 1 ) { throw "To user '$ToUser' is a group." }
                if ( !$toUserObjects[0].HasFHAccountEnabled ) { throw "To user '$ToUser' is disabled." }

            }
        }

        $fromUserSettings = (UserPreferences).GetUserWorkflowNotificationsSettings( $fromUserObjects[0].ID )
        if ( $ToUser -ne '' )
        {
            $fromUserSettings.DelegateToUser = $toUserObjects[0].ID
        }
        else
        {
            $fromUserSettings.DelegateToUser = [System.Guid]::Empty
        }
        (UserPreferences).SetUserWorkflowNotificationSettings( $fromUserObjects[0].ID, $fromUserSettings )

    } catch {
        Write-Error $_
        exit
    } finally {
        if ( !$PreserveSession )
        {
            Stop-FileHoldSession -InformationAction $InformationPreference
        }
    }
}

try {
    Export-ModuleMember -Function Set-FileHoldAutoDelegateUser                                 
} catch {
    Write-Warning "Set-FileHoldAutoDelegateUser executing directly."
}

# Load FH API functions. Assume psm1 is in same location as this script.
#    Import-Module "$PSScriptRoot\FileHoldApi.psm1" -Force
#    . "$PSScriptRoot\Get-FileHoldUserObjects.ps1"
#    Set-FileHoldAutoDelegateUser -Server 'http://rtest13' -UserId 'sysadm' -Password '12345' -SearchType ExactName -FromUser 'user one' -ToUser sysadm -InformationAction Continue -Verbose
